import argparse
import os
import subprocess
import sys

# 定义ffmpeg支持的音频文件格式后缀列表
AUDIO_EXTENSIONS = (
    '.aac', '.aiff', '.alac', '.amr', '.ape', '.au', '.flac', 
    '.m4a', '.mp3', '.ogg', '.opus', '.wav', '.wma', '.webm'
)

def check_audio_file(file_path, output_file):
    try:
        result = subprocess.run(
            ["ffmpeg", "-v", "error", "-i", file_path, "-f", "null", "-"],
            stderr=subprocess.PIPE,
            universal_newlines=True  # 如果Python版本低于3.7，应使用text=True
        )
        error_output = result.stderr
        if error_output:
            with open(output_file, "a") as f:
                f.write(f"文件损坏: {file_path}\n")
                f.write("错误详情:\n")
                f.write(error_output + "\n")
    except subprocess.CalledProcessError as e:
        with open(output_file, "a") as f:
            f.write(f"检查文件时发生错误: {e}\n")

def walk_directory(directory, output_file):
    for root, dirs, files in os.walk(directory):
        for file in files:
            if file.lower().endswith(AUDIO_EXTENSIONS):
                file_path = os.path.join(root, file)
                check_audio_file(file_path, output_file)

def main():
    # 如果没有提供任何参数，直接显示帮助信息
    if len(sys.argv) == 1:
        sys.argv.append('-h')

    parser = argparse.ArgumentParser(description="检查音频文件是否损坏")
    parser.add_argument("-i", "--input", help="需要检测的文件夹路径", required=True)
    parser.add_argument("-o", "--output", help="错误输出的文本文件路径（可选）")
    args = parser.parse_args()

    # 设置输出文件路径，如果未指定则使用脚本目录下的err.txt
    script_dir = os.path.dirname(os.path.realpath(__file__)) if __file__ else '.'
    output_file = args.output if args.output else os.path.join(script_dir, 'err.txt')

    # 遍历指定文件夹
    walk_directory(args.input, output_file)
    print(f"所有文件检查完毕，详情请查看输出文件：{output_file}")

if __name__ == "__main__":
    main()
